/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/


/* ---------------------------------------------------------------
 * File: timeout.h
 * -----------------------------------------------------------------*/

/* --------------------------------------------------------------------------
 * ABSTRACT: This module declares functions and macros to support timouts 
 *           for Best I/O.
 *           All functions support the BESTTIMEOUTS structure declared in
 *           b_io.h and are operating system independent EXCEPT:
 *           - OS must supply a function that returns an incrementing counter.
 *
 * Some of the constructs here look a little strange ... they are
 * designed to be used by WinNT/Win95 device drivers, Unix, BestX Firmware
 * as well as C-API application code...the fact that it works at all is strange !!!
 *
 * NOTES:    In accordance with the BESTTIMEOUTS structure, all timing is done
 *           in milliseconds.
 *           The meanings of all elements of the structure are identical to 
 *           those described in the WinApi help for "COMMTIMEOUTS" IF....
 *           The driver uses these functions as shown below.
 * 
 * USAGE:    A low-level driver must provide static storage for 
 *           two instances of the BESTTIMEOUTS structure; one is a default 
 *           structure which is used for initialization and if timeouts are 
 *           not set by the user.  The other is the main structure which is 
 *           used to initialize timeouts before every read.  
 *           A low-level driver must also provide a function (or IO_CTRL) 
 *           to allow a caller to set the main BESTTIMEOUTS structure.
 *           On loading the driver should copy the default to the main structure.  
 *           once only, i.e.
 *
 *        .../
 *        BESTTIMEOUTS usrTimeouts;
 *        BESTTIMEOUTS defTimeouts = NO_TIMEOUTS;   // macro below
 *        .../
 *        .../
 *        my_driver_entry(...)
 *        {
 *          usrTimeouts = defTimeouts;
 *        .../
 *        }
 *
 *        .../
 *        my_set_commtimeouts(BESTTIMEOUTS * pTimeouts)
 *        {
 *          if(pTimeouts)
 *          {
 *            usrTimeouts = *pTimeouts;
 *          }
 *          .../ any other special checking
 *        }
 *
 *
 *        High-level drivers can use the timeouts element of the handle array.
 *        This is automatically initialized on open but a dummy setting function
 *        must still be provided;
 *        .../
 *        my_set_commtimeouts(BESTTIMEOUTS * pTimeouts)
 *        {
 *          return B_E_OK;
 *        }
 *
 *
 *
 * Reading;  An example of a polled read with timeouts is....
 *        .../
 *        my_read_func(..., Int32 numbytes, Int32 * numread)
 *        {
 *          BEST_TIMER tmrInterval, tmrTotal;
 *          BestInitReadTimeouts(&tmrInterval, &tmrTotal, &usrTimeouts, numbytes);
 *          *numread = 0;
 *
 *          while( *numread < numbytes )
 *          {
 *            if(port status == DATA_RECEIVED) 
 *            {
 *              put data into buffer;
 *              ++(*numread);
 *              BestRestartIntervalTimeout(&tmrInterval); 
 *              // *** MUST reset here only after a successful read
 *            }
 *            else if(BestIsReadTimeoutDone(&tmrInterval, &tmrTotal))
 *              // *** only call BestIsReadTimeoutDone() if read unsuccessful
 *            {
 *              return NO_ERR;    
 *              // ***NOTE 1*** 
 *            }
 *          }
 *        }
 *
 *        NOTE 1; It is VERY important that you do not identify a timeout
 *              by returning an error code.  Depending on the way that
 *              the user sets up the BESTTIMEOUTS structure this may NOT
 *              be an error.  It is the application's responsibility to
 *              check "numread" against "numbytes" and determine that
 *              a timeout occurred.
 *        NOTE 2; "Immediate" timeouts may exist such that they return "done"
 *              the FIRST time that they are checked.  Thus; 
 *              *** ONLY check a timeout after an UNSUCCESSFUL READ ***
 *        NOTE 3; The interval timeout does NOT take precedence over the total
 *              timeout...both are checked in the function BestIsReadTimeoutDone
 *              but the interval timeout does not start until a character is
 *              successfully read (i.e. the first time BestRestartIntervalTimeout()
 *              is called).
 *        NOTE 4; if you have a separate routine for byte or word read then
 *              that routine should have NO knowledge of timeouts.  If the
 *              port status does not permit immediate I/O then that function
 *              should return an error code to the multi-byte calling function.
 *              Both the interval and total timeouts must be in the same 
 *              function and must both be checked at the same time.
 *
 *
 * Writing;  An example of a polled write with timeouts is....
 *        .../
 *        my_write_func(..., Int32 numbytes, Int32 * numwritten)
 *        {
 *          BEST_TIMER tmrTotal;
 *          BestInitWriteTimeout(&tmrTotal, &usrTimeouts, numbytes);
 *          *numwritten = 0;
 *
 *          while( *numwritten < numbytes )
 *          {
 *            if(port status == READY_TO_SEND) 
 *            {
 *              send data;
 *              get ack if required;
 *              ++(*numwritten);
 *            }
 *            else if(BestIsWriteTimeoutDone(&tmrTotal))
 *              return NO_ERR;    ***SEE NOTE 1 ABOVE*** 
 *          }
 *        }
 *
 *        NOTE; Write has no interval timer
 *
 *
 *
 * Looping;  A simple loop timer is accomplished thus ....
 *        .../
 *        my_do_something_func()
 *        {
 *          BEST_TIMER tmr;
 *          BestStartTimeout(100, &tmr);  *** 100 mS ***
 *
 *          while( ImDoingSomethingAndNotSucceeding() )
 *          {
 *            if(BestIsTimeoutDone(&tmr)) 
 *            {
 *              return B_E_TIMEOUT;
 *            }
 *          }
 *          return B_E_OK;
 *        }
 *
 * See additional notes in the WinApi help for "COMMTIMEOUTS"
 *
 * -------------------------------------------------------------------------- */

#ifndef TIMEOUTS_H__INCLUDED
#define TIMEOUTS_H__INCLUDED

#ifdef BESTX_COMPILE
#include <xtypedef.h>

typedef bx_int32 b_int32;
typedef bx_int8 b_int8;
typedef bx_bool b_bool;
typedef bx_int16 b_int16;
#endif



/******************************************************************************/
/* FIRST; Put all additional OS platform defines here !!!
 * Makefile defines;
 * NT driver; (DEVL=1) or NT_DRIVER
 * 95 driver; _VXD_
 * Windows;   WIN32 or _WIN32
 * Unix;      UNIX or UNIX_DBG
 * Firmware;  BEST_FIRMWARE
 */
#if defined(DEVL) && (DEVL == 1) && !defined(NT_DRIVER)
# define NT_DRIVER  1
#endif  /* (DEVL == 1) */

#if defined(NT_DRIVER) || defined(_VXD_) 
# define WIN_DEV_DRIVER 1
#endif

#if defined(UNIX) || defined(UNIX_DBG)
# define UNIX_COMPILE 1
#endif

#if defined(LINUX)
# define LINUX_COMPILE 1
# ifdef __KERNEL__
#  include <linux/time.h>
# else
#  include <time.h>
#  include <sys/time.h>
# endif
#endif

#if defined(WIN32) || defined(_WIN32) || defined(WIN64) || defined(_WIN64)
# define WIN32_COMPILE 1
#endif


/******************************************************************************/
/* all other include files here */
#if defined(NT_DRIVER)
# if (_WIN32_WINNT>=0x0501)
#  include <wdm.h>
# else
#  include <ntddk.h>
# endif

#else   /* NT_DRIVER */

/* all except NT drivers */

  /* memset() is in one or both of these */

#ifdef __KERNEL__
# include <linux/slab.h>
# define malloc(len) kmalloc(len,GFP_KERNEL)
#elif defined DOSW32
#include <malloc.h>
#else
#include <memory.h>
#endif

#ifdef __KERNEL__
# include <linux/string.h>
#else
#include <string.h>
#endif

# if defined(_VXD_)
#   include <basedef.h>
#   include <vmm.h>
#   include <time.h>

# elif defined(BEST_FIRMWARE)
#   include <process.h>
#   include "cpu.h"

# elif defined(LINUX_COMPILE)

# else                      /* all others */
#   include <time.h>

#   ifdef WIN32_COMPILE
      /* the pragmas allow a level-4 compile without going crazy */
#     pragma warning( disable : 4142 4201 4214 4115 4514 )
#     include <windows.h>
#     include <mmsystem.h>    /* REMINDER: link in winmm.lib */
#     pragma warning( default : 4142 4201 4214 4115 )
#   endif

# endif /* _VXD_ */

#endif  /* NT_DRIVER */


/******************************************************************************/
/* our version of the Win32 MAXDWORD */

#define B_INT32MAX    0xffffffffUL


/******************************************************************************/
/* b_ticktype is our version of the OS's data type for a "tick". */

#if defined(NT_DRIVER)
  typedef LARGE_INTEGER b_ticktype;

#elif defined(_VXD_)
  typedef unsigned long b_ticktype;

#elif defined(WIN32_COMPILE)
  typedef unsigned long b_ticktype;

#elif defined(BEST_FIRMWARE)
  typedef unsigned long b_ticktype;

#else    /* the ANSI standard clock() returns clock_t (== long) */
  typedef clock_t b_ticktype;

#endif  /* NT_DRIVER */


/******************************************************************************/
/* We need to be able to add and compare b_ticktype's. */

#if defined(NT_DRIVER)

# define _ADD(x,y) RtlLargeIntegerAdd((x), (y))
# define _ISGT(x,y) RtlLargeIntegerGreaterThan((x), (y))

#else   /* NT_DRIVER */

# define _ADD(x,y) ((x) + (y))
# define _ISGT(x,y) ((x) > (y))

#endif  /* NT_DRIVER */


/******************************************************************************/
/* The OS's incrementing counter and tick functions.
 * - TICKS_PER_SEC    That the OS delivers.
 * - ULONG_2_TICK     Conversion from unsigned long to b_ticktype
 * - CLEAR_TIMER      A mechanism for filling a BEST_TIMER structure with 0's.
 */

#if defined(NT_DRIVER)

# define TICKS_PER_SEC   ((10000000UL) / KeQueryTimeIncrement())
# ifndef _IA64_
#  define ULONG_2_TICK(x) RtlConvertUlongToLargeInteger((x))
# endif
# define CLEAR_TIMER(pDest) RtlZeroMemory((pDest), sizeof(BEST_TIMER))

#else   /* NT_DRIVER */

/* CLOCKS_PER_SEC is strict ANSI so we make any adjustments first; */
# ifdef BEST_FIRMWARE
#   define CLOCKS_PER_SEC   SECOND_TICKS
# endif /* BEST_FIRMWARE */

# define TICKS_PER_SEC   CLOCKS_PER_SEC
# define ULONG_2_TICK(x)  (b_ticktype)((x))
# define CLEAR_TIMER(pDest) memset((pDest), 0, sizeof(BEST_TIMER))

#endif  /* NT_DRIVER */


/******************************************************************************/
/* - GET_TICK_COUNT   The OS's function call to get the current ticks 
 * Note; the return datatype of the function used in GET_TICK_COUNT 
 * must be b_ticktype. 
 */

#if defined(NT_DRIVER)
#  define GET_TICK_COUNT(ptr_b_ticktype)  KeQueryTickCount((ptr_b_ticktype))

#elif defined(_VXD_)
   DWORD Get_System_Time(void); /* in vmm.inc */
#  define GET_TICK_COUNT(ptr_b_ticktype)  (*(ptr_b_ticktype) = Get_System_Time())

#elif defined(BEST_FIRMWARE)
#  define GET_TICK_COUNT(ptr_b_ticktype)  (*(ptr_b_ticktype) = CPUTickCount())

#elif defined(WIN32_COMPILE)
#  define GET_TICK_COUNT(ptr_b_ticktype)  (*(ptr_b_ticktype) = timeGetTime())

#elif defined(UNIX_COMPILE)
#  define GET_TICK_COUNT(ptr_b_ticktype) \
      struct timeval tv;      \
      struct timezone tz;     \
      gettimeofday(&tv, &tz); \
      *(ptr_b_ticktype) = (b_ticktype)((tv.tv_sec * 1e6) + tv.tv_usec)

#elif defined(LINUX_COMPILE)
#  define GET_TICK_COUNT(ptr_b_ticktype) \
      struct timeval tv;      \
      do_gettimeofday(&tv); \
      *(ptr_b_ticktype) = (b_ticktype)((tv.tv_sec * 1e6) + tv.tv_usec)

#elif defined(DOSW32)
   /* Watcom clock function call sometimes returns invalid value and also
   *  causes reset of the BIOS timer tic at 40:6c to value when program was
   *  started. Happens only after reboot, when run with DOS4GW DOS extender.
   *  Suspect some sort of initialization problem because it goes away after
   *  running compiler. Unclear where the problem lies (Watcom, DOS4GW or
   *  both). Work around inplemented is to not call clock anywhere in the
   *  code and to use BIOS timer tic directly. <dkush 11/01/2002>
   */
#  define GET_TICK_COUNT(ptr_b_ticktype)  (*(ptr_b_ticktype) = \
          *((unsigned long far *) (((unsigned short)0x40):>((void __near *) 0x6c))))

#else  /* strict ANSI platforms */
       /* note that clock() is VERY SLOW compared to timeGetTime() */
#  define GET_TICK_COUNT(ptr_b_ticktype)  (*(ptr_b_ticktype) = clock())

#endif  /* NT_DRIVER */



/* --------------------------------------------------------------------------
 * The user interface to timeouts....AT LAST !!!
 * -------------------------------------------------------------------------- */


/* --------------------------------------------------------------------------
 * The BESTTIMEOUTS structure allows an application to specify sophisticated
 * timeout handling and evaluation.  Read timeouts are separately 
 * set from Write timeouts and total timeouts depend on the number of 
 * characters expected.  A Read Interval timeout determines the timeout 
 * between 2 consecutive characters in a stream (after the first 
 * character is received).
 * -------------------------------------------------------------------------- */

#ifndef BESTTIMEOUTS_DECL /* make drivers/firmware independent of typedefs.h */

#define BESTTIMEOUTS_DECL 1

  typedef int b_bool;
  typedef unsigned char b_int8;
  typedef unsigned short b_int16;
  typedef unsigned long b_int32;

  typedef struct _BESTTIMEOUTS {
  b_int32 ReadIntervalTimeout;          /* Maximum time between read chars. */
  b_int32 ReadTotalTimeoutMultiplier;   /* Multiplier of characters.        */
  b_int32 ReadTotalTimeoutConstant;     /* Constant in milliseconds.        */
  b_int32 WriteTotalTimeoutMultiplier;  /* Multiplier of characters.        */
  b_int32 WriteTotalTimeoutConstant;    /* Constant in milliseconds.        */

} BESTTIMEOUTS, *LPBESTTIMEOUTS;

#endif /* BESTTIMEOUTS_DECL */

/* --------------------------------------------------------------------------
 * The BEST_TIMER structure is the basic data type for all timeouts, 
 * both simple and complicated, whether using the BESTTIMEOUTS structure or
 * not.  One of these must exist for each timeout used.
 * -------------------------------------------------------------------------- */

typedef struct _BEST_TIMER
{
  b_ticktype StartTicks;
  b_ticktype EndTicks;
  b_int32 IntervalInMs;    /* here for speed */
  b_int8 fWrappedAround;
  b_int8 fIsInfinite;
  b_int8 fIsImmediate;
  b_int8 fWaitForRestart;  /* interval timing begins after 1st char */

} BEST_TIMER, *PBEST_TIMER;


/* --------------------------------------------------------------------------
 * These functions are for use with general purpose (simple) timeouts
 * and are used instead of timing loops.
 * -------------------------------------------------------------------------- */

void BestStartTimeout(b_int32 IntervalInMs, PBEST_TIMER pBTS);
  b_bool BestIsTimeoutDone(PBEST_TIMER pBTS);


/* --------------------------------------------------------------------------
 * These functions are used for more sophisticated timeouts ... generally
 * in an I/O context and always while using the BESTTIMEOUTS structure.  
 * They are mostly implemented in drivers and allow for quite complex 
 * behavior, especially while reading.
 * -------------------------------------------------------------------------- */

void BestInitReadTimeouts(PBEST_TIMER pTmrInterval, 
                          PBEST_TIMER pTmrTotal, 
                          LPBESTTIMEOUTS pUsrTimeouts, 
  b_int32 NumBytes);

void BestInitWriteTimeout(PBEST_TIMER pTmrTotal, 
                          LPBESTTIMEOUTS pCommTimeouts,
  b_int32 NumBytes);

void BestRestartIntervalTimeout(PBEST_TIMER pTmrInterval);

  b_bool BestIsReadTimeoutDone(PBEST_TIMER pInterval, PBEST_TIMER pTotal);
  b_bool BestIsWriteTimeoutDone(PBEST_TIMER pBTS);

void BestSetCommToStruct(LPBESTTIMEOUTS pCommTimeouts, 
  b_int32 ReadIntervalTimeout, 
  b_int32 ReadTotalTimeoutMultiplier, 
  b_int32 ReadTotalTimeoutConstant,
  b_int32 WriteTotalTimeoutMultiplier,
  b_int32 WriteTotalTimeoutConstant);


/* --------------------------------------------------------------------------
 * special initialization of a BESTTIMEOUTS structure
 * -------------------------------------------------------------------------- */

/* infinite (no timeouts used) */
#define NO_TIMEOUTS           {0,0,0,0,0}

/* immediate (timeouts == 0 mS) */
#define IMM_READ_TIMEOUT      {B_INT32MAX,0,0,0,0}
#define IMM_WRITE_TIMEOUT     {0,0,0,B_INT32MAX,0}
#define IMM_IO_TIMEOUTS       {B_INT32MAX,0,0,B_INT32MAX,0}


#endif /* _TIMEOUTS_H_ */
